package com.hhwebservice.utils.file;


import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.Reader;
import java.io.Writer;

/**
 * Methoden zur Manipulation und Verarbeitung von Streams.
 *
 */

public class Streams 
{                                                                                         
	    private static byte[] $buffer = new byte[2048];

	    public Streams()
	    {
	        super();
	    }

	    /**
	     * Kopiert einen Input-Stream blockweise in einen Output-Stream.
	     *
	     * Der Block hat einen Gre von 2048 Bytes.
	     *
	     * @param  source           Die Quelle
	     * @param  destination      Das Ziel
	     * @param  autoClose        Die Streams sollen nach dem Kopieren geschlossen werden
	     * @return                  Die Anzahl der kopierten Bytes, oder ein negativer Wert, wenn ein Fehler aufgetreten ist
	     */

	    public synchronized static int copy(InputStream source,OutputStream destination,boolean autoClose)
	    {
	        int sum = 0;

	        try
	        {
	            int n;
	            while ((n = source.read($buffer)) >= 0)
	            {
	                sum += n;
	                destination.write($buffer,0,n);
	            }
	        }
	        catch (Exception e)
	        {
	            sum = (sum == 0)? Integer.MIN_VALUE: -sum;
	        }
	        finally
	        {
	            if (autoClose)
	            {
	                closeQuietly(source);
	                closeQuietly(destination);
	            }
	        }

	        return sum;
	    }


	    /**
	     * Liefert den Inhalt eines Streams.
	     *
	     * @param  stream           Der Stream
	     * @param  autoClose        Der Stream sollen nach dem Lesen geschlossen werden
	     * @return                  Der Inhalt der Datei
	     * @throws IOException      Die Datei konnte nicht gelesen werden
	     */

	    public static synchronized byte[] getBytes(InputStream stream,boolean autoClose) throws IOException
	    {
	        int length = $buffer.length;
	        byte[] result = new byte[length];

	        try
	        {
	            int offset = 0;
	            
	            int n;
	            while ((n = stream.read($buffer)) >= 0)
	            {
	                if (length < offset + n)
	                {
	                    while (length < offset + n)
	                        length <<= 1;
	                    
	                    byte[] buffer = new byte[length];
	                    System.arraycopy(result,0,buffer,0,offset);
	                    
	                    result = buffer;
	                    buffer = null; // Explicitly inform the garbage collector about this array
	                }
	                
	                System.arraycopy($buffer,0,result,offset,n);
	                offset += n;
	            }
	            
	            if (offset < length)
	            {
	                byte[] buffer = new byte[offset];
	                System.arraycopy(result,0,buffer,0,offset);
	                
	                result = buffer;
	                buffer = null; // Explicitly inform the garbage collector about this array
	            }
	            
	            return result;
	        }
	        finally
	        {
	            if (autoClose)
	                Streams.closeQuietly(stream);
	        }
	    }


	    /**
	     * Liefert den Inhalt eines Streams als String.
	     *
	     * @param  stream           Der Stream
	     * @param  autoClose        Der Stream sollen nach dem Lesen geschlossen werden
	     * @return                  Der Inhalt der Datei als String
	     * @throws IOException      Die Datei konnte nicht gelesen werden
	     */

	    public static String getContent(InputStream stream,boolean autoClose) throws IOException
	    {
	        return new String(getBytes(stream,autoClose));
	    }
	    
	    
	    /**
	     * Schliesst einen Input-Stream, ohne dass I/O-Fehler bercksichtigt werden.
	     * 
	     * @param  inputStream      Der Input-Stream
	     */
	    
	    public static void closeQuietly(InputStream inputStream)
	    {
	        try
	        {
	            if (inputStream != null)
	                inputStream.close();
	        }
	        catch (IOException e)
	        {
	            // Keine Aktion ntig
	        }
	    }
	    
	    
	    /**
	     * Schliesst einen Output-Stream, ohne dass I/O-Fehler bercksichtigt werden.
	     * 
	     * @param  outputStream     Der Output-Stream
	     */
	    
	    public static void closeQuietly(OutputStream outputStream)
	    {
	        try
	        {
	            if (outputStream != null)
	                outputStream.close();
	        }
	        catch (IOException e)
	        {
	            // Keine Aktion ntig
	        }
	    }
	    
	    
	    /**
	     * Schliesst einen Reader, ohne dass I/O-Fehler bercksichtigt werden.
	     * 
	     * @param  reader           Der Reader
	     */
	    
	    public static void closeQuietly(Reader reader)
	    {
	        try
	        {
	            if (reader != null)
	                reader.close();
	        }
	        catch (IOException e)
	        {
	            // Keine Aktion ntig
	        }
	    }
	    
	    
	    /**
	     * Schliesst einen Writer, ohne dass I/O-Fehler bercksichtigt werden.
	     * 
	     * @param  writer           Der Writer
	     */
	    
	    public static void closeQuietly(Writer writer)
	    {
	        try
	        {
	            if (writer != null)
	                writer.close();
	        }
	        catch (IOException e)
	        {
	            // Keine Aktion ntig
	        }
	    }


}
